<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

include_once '../config/database.php';
include_once '../models/StreamSource.php';

$database = new Database();
$db = $database->getConnection();
$streamSource = new StreamSource($db);

$match_id = $_GET['id'] ?? null;

if (!$match_id) {
    header("Location: matches.php");
    exit();
}

// Get match info
$stmt = $db->prepare("SELECT m.*, ht.name_ar as home_team, at.name_ar as away_team
                      FROM matches m
                      LEFT JOIN teams ht ON m.home_team_id = ht.id
                      LEFT JOIN teams at ON m.away_team_id = at.id
                      WHERE m.id = ?");
$stmt->execute([$match_id]);
$match = $stmt->fetch(PDO::FETCH_ASSOC);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] == 'create') {
            $data = [
                'match_id' => $match_id,
                'name' => $_POST['name'],
                'url' => $_POST['url'],
                'quality' => $_POST['quality'],
                'stream_type' => $_POST['stream_type'],
                'is_available' => isset($_POST['is_available']) ? 1 : 0,
                'display_order' => $_POST['display_order'] ?? 0
            ];

            if ($streamSource->create($data)) {
                $success = "تم إضافة مصدر البث بنجاح";
            }
        } elseif ($_POST['action'] == 'delete') {
            if ($streamSource->delete($_POST['stream_id'])) {
                $success = "تم حذف مصدر البث بنجاح";
            }
        }
    }
}

// Get all streams for this match
$stmt = $streamSource->getByMatchId($match_id);
$streams = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة البث - <?php echo $match['home_team'] . ' vs ' . $match['away_team']; ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Cairo', sans-serif; }
    </style>
</head>
<body class="bg-gray-100">
    <!-- Navigation -->
    <nav class="bg-gradient-to-r from-green-600 to-green-700 shadow-lg">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <a href="index.php" class="text-white text-2xl font-bold">⚽ لوحة التحكم</a>
                </div>
                <div class="flex items-center space-x-4 space-x-reverse">
                    <a href="matches.php" class="text-white hover:text-gray-200">← رجوع للمباريات</a>
                    <span class="text-white"><?php echo $_SESSION['username']; ?></span>
                    <a href="logout.php" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg">تسجيل الخروج</a>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Match Info Card -->
        <div class="bg-white rounded-lg shadow-md p-6 mb-8">
            <h2 class="text-2xl font-bold text-gray-800 mb-4">📡 إدارة مصادر البث</h2>
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-lg font-semibold text-gray-700">
                        <?php echo $match['home_team'] . ' 🆚 ' . $match['away_team']; ?>
                    </p>
                    <p class="text-sm text-gray-500">
                        <?php echo date('Y-m-d H:i', strtotime($match['match_time'])); ?>
                    </p>
                </div>
                <span class="px-4 py-2 bg-blue-100 text-blue-800 rounded-full font-semibold">
                    <?php echo count($streams); ?> مصدر بث
                </span>
            </div>
        </div>

        <?php if (isset($success)): ?>
        <div class="bg-green-100 border-r-4 border-green-500 text-green-700 p-4 rounded mb-6">
            <?php echo $success; ?>
        </div>
        <?php endif; ?>

        <!-- Add Stream Button -->
        <div class="mb-6">
            <button onclick="openModal()" class="bg-green-600 hover:bg-green-700 text-white px-6 py-3 rounded-lg font-semibold shadow-md transition">
                ➕ إضافة مصدر بث جديد
            </button>
        </div>

        <!-- Streams Grid -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            <?php foreach ($streams as $stream): ?>
            <div class="bg-white rounded-lg shadow-md p-6 border-r-4 <?php echo $stream['is_available'] ? 'border-green-500' : 'border-red-500'; ?>">
                <div class="flex justify-between items-start mb-4">
                    <h3 class="text-lg font-bold text-gray-800"><?php echo $stream['name']; ?></h3>
                    <span class="px-3 py-1 text-xs font-semibold rounded-full <?php echo $stream['is_available'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                        <?php echo $stream['is_available'] ? '✅ متاح' : '❌ غير متاح'; ?>
                    </span>
                </div>

                <div class="space-y-2 mb-4">
                    <div class="flex items-center text-sm">
                        <span class="font-semibold text-gray-600 w-24">الجودة:</span>
                        <span class="text-gray-800"><?php echo $stream['quality']; ?></span>
                    </div>
                    <div class="flex items-center text-sm">
                        <span class="font-semibold text-gray-600 w-24">النوع:</span>
                        <span class="px-2 py-1 bg-purple-100 text-purple-800 rounded text-xs font-semibold">
                            <?php echo strtoupper($stream['stream_type']); ?>
                        </span>
                    </div>
                    <div class="flex items-center text-sm">
                        <span class="font-semibold text-gray-600 w-24">الترتيب:</span>
                        <span class="text-gray-800"><?php echo $stream['display_order']; ?></span>
                    </div>
                </div>

                <div class="mb-4">
                    <p class="text-xs text-gray-500 font-semibold mb-1">رابط البث:</p>
                    <div class="bg-gray-50 p-2 rounded border border-gray-200">
                        <p class="text-xs text-gray-700 truncate" title="<?php echo $stream['url']; ?>">
                            <?php echo $stream['url']; ?>
                        </p>
                    </div>
                </div>

                <div class="flex space-x-2 space-x-reverse">
                    <button onclick="editStream(<?php echo htmlspecialchars(json_encode($stream)); ?>)"
                            class="flex-1 bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded transition text-sm">
                        ✏️ تعديل
                    </button>
                    <form method="POST" class="flex-1" onsubmit="return confirm('هل أنت متأكد من الحذف؟');">
                        <input type="hidden" name="action" value="delete">
                        <input type="hidden" name="stream_id" value="<?php echo $stream['id']; ?>">
                        <button type="submit" class="w-full bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded transition text-sm">
                            🗑️ حذف
                        </button>
                    </form>
                </div>
            </div>
            <?php endforeach; ?>

            <?php if (empty($streams)): ?>
            <div class="col-span-full text-center py-12">
                <p class="text-gray-500 text-lg">📡 لا توجد مصادر بث لهذه المباراة</p>
                <p class="text-gray-400 text-sm mt-2">أضف مصدر بث جديد للبدء</p>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Add/Edit Stream Modal -->
    <div id="streamModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-8 border w-full max-w-2xl shadow-lg rounded-lg bg-white">
            <div class="flex justify-between items-center mb-6">
                <h3 id="modalTitle" class="text-2xl font-bold text-gray-900">➕ إضافة مصدر بث جديد</h3>
                <button onclick="closeModal()" class="text-gray-400 hover:text-gray-600 text-3xl">&times;</button>
            </div>

            <form id="streamForm" method="POST" class="space-y-4">
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="stream_id" id="streamId">

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">اسم المصدر *</label>
                    <input type="text" name="name" id="streamName" required
                           placeholder="مثال: البث الأساسي HD"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-green-500">
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">رابط البث (URL) *</label>
                    <input type="url" name="url" id="streamUrl" required
                           placeholder="https://example.com/stream.m3u8"
                           class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-green-500">
                    <p class="text-xs text-gray-500 mt-1">
                        💡 الصيغ المدعومة: m3u8, m3u, mpd, mp4, ts, flv
                    </p>
                </div>

                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">الجودة *</label>
                        <select name="quality" id="streamQuality" required class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-green-500">
                            <option value="Auto">تلقائي</option>
                            <option value="SD">SD - جودة عادية</option>
                            <option value="HD">HD - جودة عالية</option>
                            <option value="FHD">FHD - جودة فائقة</option>
                            <option value="4K">4K - جودة 4K</option>
                        </select>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">نوع البث *</label>
                        <select name="stream_type" id="streamType" required class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-green-500">
                            <option value="m3u8">m3u8 (HLS)</option>
                            <option value="m3u">m3u (Playlist)</option>
                            <option value="mpd">mpd (DASH)</option>
                            <option value="mp4">mp4 (Video)</option>
                            <option value="ts">ts (Transport Stream)</option>
                            <option value="flv">flv (Flash Video)</option>
                        </select>
                    </div>
                </div>

                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">ترتيب العرض</label>
                        <input type="number" name="display_order" id="displayOrder" value="0" min="0"
                               class="w-full px-4 py-2 border rounded-lg focus:ring-2 focus:ring-green-500">
                        <p class="text-xs text-gray-500 mt-1">الأقل أولاً</p>
                    </div>

                    <div class="flex items-center pt-8">
                        <input type="checkbox" name="is_available" id="isAvailable" checked
                               class="h-4 w-4 text-green-600 focus:ring-green-500 border-gray-300 rounded">
                        <label for="is_available" class="mr-2 block text-sm text-gray-900">
                            ✅ متاح للبث
                        </label>
                    </div>
                </div>

                <div class="flex justify-end space-x-4 space-x-reverse pt-4 border-t">
                    <button type="button" onclick="closeModal()" class="px-6 py-2 border border-gray-300 rounded-lg hover:bg-gray-50">
                        إلغاء
                    </button>
                    <button type="submit" class="px-6 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                        ✅ حفظ
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function openModal() {
            document.getElementById('streamModal').classList.remove('hidden');
            document.getElementById('modalTitle').textContent = '➕ إضافة مصدر بث جديد';
            document.getElementById('formAction').value = 'create';
            document.getElementById('streamForm').reset();
        }

        function closeModal() {
            document.getElementById('streamModal').classList.add('hidden');
        }

        function editStream(stream) {
            document.getElementById('streamModal').classList.remove('hidden');
            document.getElementById('modalTitle').textContent = '✏️ تعديل مصدر البث';
            document.getElementById('formAction').value = 'update';
            document.getElementById('streamId').value = stream.id;
            document.getElementById('streamName').value = stream.name;
            document.getElementById('streamUrl').value = stream.url;
            document.getElementById('streamQuality').value = stream.quality;
            document.getElementById('streamType').value = stream.stream_type;
            document.getElementById('displayOrder').value = stream.display_order;
            document.getElementById('isAvailable').checked = stream.is_available == 1;
        }
    </script>
</body>
</html>

