# 🎯 API إدارة تطبيق كرة القدم

## 📚 نظام إدارة كامل باللغة العربية

### ✨ المميزات الرئيسية

#### 🎛️ **لوحة تحكم احترافية**
- ✅ واجهة عربية كاملة مع Tailwind CSS
- ✅ تصميم عصري وسريع الاستجابة
- ✅ إحصائيات فورية ولايف
- ✅ إدارة كاملة لجميع عناصر التطبيق

#### ⚽ **إدارة المباريات**
- إضافة وتعديل وحذف المباريات
- تتبع النتائج المباشرة
- إدارة أحداث المباراة (أهداف، بطاقات، تبديلات)
- ربط المباريات بالدوريات والفرق

#### 📡 **إدارة البث المباشر**
- إضافة مصادر بث متعددة لكل مباراة
- دعم جميع صيغ IPTV (m3u8, mpd, mp4...)
- تحديد الجودة لكل مصدر
- ترتيب المصادر حسب الأولوية
- تفعيل/تعطيل مصادر البث

#### 🏆 **إدارة الدوريات والفرق**
- إدارة الدوريات المختلفة
- إضافة الفرق والشعارات
- ربط الفرق بالمباريات

---

## 🛠️ التثبيت والإعداد

### المتطلبات
```
- PHP 7.4+
- MySQL 5.7+
- Apache/Nginx
- XAMPP أو WAMP (للتطوير المحلي)
```

### خطوات التثبيت

#### 1️⃣ **إعداد قاعدة البيانات**

```bash
# افتح phpMyAdmin أو MySQL CLI
# قم بتشغيل ملف schema.sql
```

```sql
# أو من Terminal
mysql -u root -p < api/database/schema.sql
```

#### 2️⃣ **تكوين الاتصال بقاعدة البيانات**

افتح ملف `api/config/database.php` وعدل البيانات:

```php
private $host = "localhost";
private $db_name = "football_app";
private $username = "root";
private $password = "YOUR_PASSWORD"; // ضع كلمة المرور هنا
```

#### 3️⃣ **رفع الملفات**

```bash
# انسخ مجلد api إلى مجلد الويب الخاص بك
# مثال في XAMPP:
C:\xampp\htdocs\football_app\
```

#### 4️⃣ **تسجيل الدخول**

```
URL: http://localhost/football_app/api/admin/login.php
اسم المستخدم: admin
كلمة المرور: admin123
```

---

## 📂 هيكل المشروع

```
api/
├── config/
│   └── database.php              # إعدادات قاعدة البيانات
├── database/
│   └── schema.sql                # جداول قاعدة البيانات
├── models/
│   ├── Match.php                 # نموذج المباريات
│   └── StreamSource.php          # نموذج مصادر البث
├── matches/
│   └── read.php                  # API للحصول على المباريات
├── admin/
│   ├── index.php                 # لوحة التحكم الرئيسية
│   ├── login.php                 # صفحة تسجيل الدخول
│   ├── logout.php                # تسجيل الخروج
│   ├── matches.php               # إدارة المباريات
│   └── match_streams.php         # إدارة مصادر البث
└── README.md                     # هذا الملف
```

---

## 🔌 واجهات API البرمجية

### 1. **الحصول على جميع المباريات**

```http
GET /api/matches/read.php

Query Parameters:
- status: live|upcoming|finished (اختياري)
- limit: عدد النتائج (افتراضي: 50)
- offset: البداية (افتراضي: 0)
```

**مثال على الطلب:**
```bash
curl http://localhost/football_app/api/matches/read.php?status=live
```

**مثال على الاستجابة:**
```json
{
  "success": true,
  "count": 2,
  "data": [
    {
      "id": "1",
      "homeTeam": "الهلال",
      "awayTeam": "النصر",
      "homeScore": 2,
      "awayScore": 1,
      "status": "live",
      "league": "دوري روشن السعودي",
      "matchTime": "2025-11-20 20:00:00",
      "currentMinute": 65,
      "homeTeamLogo": null,
      "awayTeamLogo": null,
      "stadium": "ملعب الملك فهد",
      "referee": "محمد العبدلي",
      "events": [
        {
          "type": "goal",
          "team": "home",
          "minute": 23,
          "player": "سالم الدوسري",
          "assistPlayer": "مالكوم"
        }
      ],
      "streamSources": [
        {
          "id": "1",
          "name": "البث الأساسي",
          "url": "https://example.com/stream.m3u8",
          "quality": "FHD",
          "type": "m3u8",
          "isAvailable": true
        }
      ]
    }
  ]
}
```

### 2. **الحصول على مباراة واحدة**

```http
GET /api/matches/read_single.php?id=1
```

---

## 🎨 لوحة التحكم

### الصفحات المتاحة

#### 1. **الرئيسية** (`index.php`)
- إحصائيات عامة
- عدد المباريات المباشرة
- عدد المباريات القادمة
- إجمالي الفرق والدوريات
- جدول آخر المباريات

#### 2. **إدارة المباريات** (`matches.php`)
- عرض جميع المباريات
- إضافة مباراة جديدة
- تعديل بيانات المباراة
- حذف المباراة
- إدارة مصادر البث

#### 3. **إدارة البث** (`match_streams.php`)
- عرض مصادر البث لمباراة معينة
- إضافة مصدر بث جديد
- تعديل معلومات المصدر
- حذف المصدر
- تفعيل/تعطيل المصدر

---

## 📊 قاعدة البيانات

### الجداول الرئيسية

#### 1. **users** - المستخدمين
```sql
- id
- username
- password (مشفرة)
- email
- role (admin/moderator/user)
```

#### 2. **leagues** - الدوريات
```sql
- id
- name (الاسم بالإنجليزية)
- name_ar (الاسم بالعربية)
- country
- logo_url
- is_active
```

#### 3. **teams** - الفرق
```sql
- id
- name
- name_ar
- logo_url
- stadium
- is_active
```

#### 4. **matches** - المباريات
```sql
- id
- league_id
- home_team_id
- away_team_id
- home_score
- away_score
- status (upcoming/live/finished)
- match_time
- current_minute
- stadium
- referee
```

#### 5. **match_events** - أحداث المباراة
```sql
- id
- match_id
- event_type (goal/yellowCard/redCard/substitution)
- team_side (home/away)
- minute
- player_name
- assist_player_name
```

#### 6. **stream_sources** - مصادر البث
```sql
- id
- match_id
- name
- url
- quality (SD/HD/FHD/4K)
- stream_type (m3u8/mpd/mp4...)
- is_available
- display_order
```

---

## 🔐 الأمان

### إجراءات الأمان المطبقة

✅ تشفير كلمات المرور (bcrypt)
✅ جلسات PHP آمنة
✅ التحقق من تسجيل الدخول في كل صفحة
✅ PDO Prepared Statements (حماية من SQL Injection)
✅ CORS Headers

### تغيير كلمة المرور

```php
// استخدم هذا الكود لتوليد كلمة مرور جديدة
$password = "your_new_password";
$hashed = password_hash($password, PASSWORD_DEFAULT);
echo $hashed;

// ثم حدّث قاعدة البيانات
UPDATE users SET password = '$hashed' WHERE username = 'admin';
```

---

## 🚀 الربط مع تطبيق Flutter

### في تطبيق Flutter

```dart
// تحديث URL الـ API في match_provider.dart
final response = await http.get(
  Uri.parse('http://YOUR_SERVER_IP/football_app/api/matches/read.php?status=live'),
);
```

### مثال كامل

```dart
Future<void> loadMatches() async {
  try {
    final response = await http.get(
      Uri.parse('http://192.168.1.100/football_app/api/matches/read.php'),
    );

    if (response.statusCode == 200) {
      final data = json.decode(response.body);
      if (data['success']) {
        final matches = (data['data'] as List)
            .map((json) => Match.fromJson(json))
            .toList();
        // استخدم البيانات
      }
    }
  } catch (e) {
    print('Error: $e');
  }
}
```

---

## 📱 التكامل مع التطبيق

### خطوات الربط

1. **تأكد من تشغيل الخادم المحلي**
```bash
# XAMPP
افتح XAMPP Control Panel
شغّل Apache و MySQL
```

2. **اختبر API**
```bash
http://localhost/football_app/api/matches/read.php
```

3. **حدّث Flutter App**
```dart
// في services/api_service.dart
class ApiService {
  static const String baseUrl = 'http://YOUR_IP/football_app/api';
  
  static Future<List<Match>> getMatches({String? status}) async {
    final response = await http.get(
      Uri.parse('$baseUrl/matches/read.php?status=${status ?? ''}'),
    );
    
    if (response.statusCode == 200) {
      final data = json.decode(response.body);
      return (data['data'] as List)
          .map((json) => Match.fromJson(json))
          .toList();
    }
    throw Exception('Failed to load matches');
  }
}
```

---

## 🎯 استخدام لوحة التحكم

### إضافة مباراة مع بث مباشر

1. **سجّل الدخول** إلى لوحة التحكم
2. اذهب إلى **"إدارة المباريات"**
3. اضغط **"إضافة مباراة جديدة"**
4. املأ البيانات:
   - اختر الدوري
   - اختر الفريقين
   - حدد وقت المباراة
   - اختر الحالة (مباشر/قادمة/منتهية)
5. احفظ المباراة
6. اضغط **"📡 البث"** لإضافة مصادر البث
7. أضف رابط البث:
   - اسم المصدر: "البث الأساسي"
   - URL: `https://your-stream.com/live.m3u8`
   - الجودة: FHD
   - النوع: m3u8
8. احفظ المصدر

---

## 🐛 استكشاف الأخطاء

### مشكلة: لا يمكن الاتصال بقاعدة البيانات

```
✅ تأكد من تشغيل MySQL
✅ تحقق من بيانات الاتصال في database.php
✅ تأكد من إنشاء قاعدة البيانات
```

### مشكلة: خطأ 404 عند فتح الصفحات

```
✅ تأكد من رفع الملفات في المسار الصحيح
✅ تحقق من URL في المتصفح
✅ تأكد من تشغيل Apache
```

### مشكلة: لا تظهر البيانات في API

```
✅ تحقق من وجود بيانات في قاعدة البيانات
✅ افتح URL الـ API مباشرة في المتصفح
✅ تحقق من console الخادم
```

---

## 📝 ملاحظات مهمة

### للإنتاج (Production)

1. **غيّر بيانات قاعدة البيانات**
2. **غيّر كلمة مرور الأدمن**
3. **استخدم HTTPS**
4. **فعّل حماية CSRF**
5. **قيّد الوصول للـ API**

### للتطوير (Development)

```php
// في database.php للـ debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

---

## 🆘 الدعم والمساعدة

### الموارد
- 📖 [PHP Documentation](https://www.php.net/docs.php)
- 🎨 [Tailwind CSS](https://tailwindcss.com/docs)
- 🐬 [MySQL Documentation](https://dev.mysql.com/doc/)

### بيانات الدخول الافتراضية
```
👤 المستخدم: admin
🔑 كلمة المرور: admin123
```

---

## ✨ التحديثات المستقبلية

- [ ] صفحة إدارة الفرق
- [ ] صفحة إدارة الدوريات  
- [ ] صفحة الإعدادات
- [ ] تقارير وإحصائيات
- [ ] إشعارات الأحداث
- [ ] رفع الصور والشعارات
- [ ] نظام الصلاحيات المتقدم
- [ ] API لتطبيقات الموبايل
- [ ] Export/Import البيانات

---

**تم التطوير بـ ❤️ للمطورين العرب**

Made with ❤️ for Arabic Developers

© 2025 Football App Management System

